<?php
// Fichier : debug-api.php
// But : tester et debugger l'API de distance

header('Content-Type: text/html; charset=utf-8');
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<h1>Debugger l'API de distance</h1>";

// Verifier si l'API est accessible
echo "<h2>Test d'accessibilite de l'API</h2>";
$apiPath = __DIR__ . '/get-distance.php';

if (file_exists($apiPath)) {
    echo "<p style='color:green'>✓ Le fichier get-distance.php existe</p>";
} else {
    echo "<p style='color:red'>✗ Le fichier get-distance.php n'existe pas!</p>";
}

// Tester l'API avec des coordonnees fixes
echo "<h2>Test avec des coordonnees fixes</h2>";

// Coordonnees de test (La Reunion)
$startLat = -21.115141;
$startLng = 55.536384;
$endLat = -21.200000;
$endLng = 55.450000;

echo "<p>Coordonnees de depart: [$startLat, $startLng]</p>";
echo "<p>Coordonnees d'arrivee: [$endLat, $endLng]</p>";

// Construire l'URL pour l'API
$host = $_SERVER['HTTP_HOST'];
$scriptPath = dirname($_SERVER['SCRIPT_NAME']);
$apiUrl = "http://$host$scriptPath/get-distance.php?start=$startLng,$startLat&end=$endLng,$endLat";

echo "<p>URL de l'API: <a href='$apiUrl' target='_blank'>$apiUrl</a></p>";

// Appeler l'API
echo "<h3>Reponse de l'API</h3>";
echo "<pre>";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
curl_setopt($ch, CURLOPT_TIMEOUT, 20);

$response = curl_exec($ch);

if (curl_errno($ch)) {
    echo "Erreur cURL: " . curl_error($ch) . "\n";
} else {
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    echo "Code HTTP: $httpCode\n\n";
    
    if ($httpCode == 200) {
        echo "Reponse brute:\n" . htmlspecialchars($response) . "\n\n";
        
        // Parser la reponse JSON
        $data = json_decode($response, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            echo "Reponse parsee:\n";
            print_r($data);
            
            // Verifier si la reponse contient une distance
            if (isset($data['features'][0]['properties']['summary']['distance'])) {
                $distance = $data['features'][0]['properties']['summary']['distance'];
                $distanceKm = $distance / 1000;
                echo "\nDistance calculee: $distanceKm km ($distance metres)\n";
            } else {
                echo "\nAucune distance trouvee dans la reponse!\n";
            }
        } else {
            echo "Erreur de parsing JSON: " . json_last_error_msg() . "\n";
        }
    } else {
        echo "Erreur HTTP: $httpCode\n";
    }
}

curl_close($ch);
echo "</pre>";

// Tester le format de la reponse
echo "<h2>Verification du format de reponse</h2>";

echo "<p>Format attendu par le JavaScript:</p>";
echo "<pre>
{
  \"type\": \"FeatureCollection\",
  \"features\": [
    {
      \"properties\": {
        \"summary\": {
          \"distance\": 12345,  // distance en metres
          \"duration\": 600     // duree en secondes
        },
        \"text_distance\": \"12,3 km\",
        \"text_duration\": \"10 minutes\"
      }
    }
  ]
}
</pre>";

// Tester l'integration JavaScript
echo "<h2>Test d'integration JavaScript</h2>";

echo "<p>Cliquez sur le bouton ci-dessous pour tester l'interception des appels API:</p>";

echo "<button id='test-button' style='padding: 10px; background-color: #3c8f44; color: white; border: none; border-radius: 5px; cursor: pointer;'>Tester l'interception API</button>";

echo "<div id='test-result' style='margin-top: 10px; padding: 10px; border: 1px solid #ccc; min-height: 100px;'></div>";

// Script de test
echo "<script>
document.getElementById('test-button').addEventListener('click', function() {
  const resultDiv = document.getElementById('test-result');
  resultDiv.innerHTML = '<p>Test en cours...</p>';
  
  // Sauvegarder la fonction fetch originale
  const originalFetch = window.fetch;
  
  // Remplacer fetch pour intercepter les appels
  window.fetch = function(url, options) {
    resultDiv.innerHTML += `<p>Appel intercepte: \${url}</p>`;
    
    // Appeler l'API
    return originalFetch(url, options).then(response => {
      // Cloner la reponse
      const responseClone = response.clone();
      
      responseClone.text().then(text => {
        resultDiv.innerHTML += `<p>Reponse recue:</p><pre>\${text}</pre>`;
        
        try {
          const data = JSON.parse(text);
          
          if (data.features && 
              data.features[0] && 
              data.features[0].properties && 
              data.features[0].properties.summary && 
              typeof data.features[0].properties.summary.distance === 'number') {
            
            const distanceKm = data.features[0].properties.summary.distance / 1000;
            resultDiv.innerHTML += `<p style='color:green'>Distance extraite: \${distanceKm.toFixed(1)} km</p>`;
          } else {
            resultDiv.innerHTML += `<p style='color:red'>Aucune distance trouvee dans la reponse!</p>`;
          }
        } catch (error) {
          resultDiv.innerHTML += `<p style='color:red'>Erreur parsing JSON: \${error.message}</p>`;
        }
      });
      
      return response;
    });
  };
  
  // Appeler l'API
  fetch('$apiUrl')
    .catch(error => {
      resultDiv.innerHTML += `<p style='color:red'>Erreur fetch: \${error.message}</p>`;
    })
    .finally(() => {
      // Restaurer la fonction fetch originale
      window.fetch = originalFetch;
    });
});
</script>";

// Verifier le fichier de log
echo "<h2>Verification du fichier de log</h2>";

$logFile = __DIR__ . '/distance_api.log';
if (file_exists($logFile)) {
    echo "<p style='color:green'>✓ Le fichier de log existe</p>";
    
    // Afficher les dernieres lignes du log
    echo "<h3>Dernieres lignes du log:</h3>";
    echo "<pre style='max-height: 300px; overflow: auto;'>";
    $log = file_get_contents($logFile);
    $lines = explode("\n", $log);
    $lastLines = array_slice($lines, -50); // Afficher les 50 dernieres lignes
    echo htmlspecialchars(implode("\n", $lastLines));
    echo "</pre>";
} else {
    echo "<p style='color:red'>✗ Le fichier de log n'existe pas!</p>";
}
